//===----------------------------------------------------------------------===//
//
// This source file is part of the Swift Argument Parser open source project
//
// Copyright (c) 2020 Apple Inc. and the Swift project authors
// Licensed under Apache License v2.0 with Runtime Library Exception
//
// See https://swift.org/LICENSE.txt for license information
//
//===----------------------------------------------------------------------===//

import ArgumentParser
import ArgumentParserTestHelpers
import XCTest

final class RawRepresentableEndToEndTests: XCTestCase {}

// MARK: -

private struct Bar: ParsableArguments {
  struct Identifier: RawRepresentable, Equatable, ExpressibleByArgument {
    var rawValue: Int
  }

  @Option() var identifier: Identifier
}

// swift-format-ignore: AlwaysUseLowerCamelCase
// https://github.com/apple/swift-argument-parser/issues/710
extension RawRepresentableEndToEndTests {
  func testParsing_SingleOption() throws {
    AssertParse(Bar.self, ["--identifier", "123"]) { bar in
      XCTAssertEqual(bar.identifier, Bar.Identifier(rawValue: 123))
    }
  }

  func testParsing_SingleOptionMultipleTimes() throws {
    AssertParse(Bar.self, ["--identifier", "123", "--identifier", "456"]) {
      bar in
      XCTAssertEqual(bar.identifier, Bar.Identifier(rawValue: 456))
    }
  }

  func testParsing_SingleOption_Fails() throws {
    XCTAssertThrowsError(try Bar.parse([]))
    XCTAssertThrowsError(try Bar.parse(["--identifier"]))
    XCTAssertThrowsError(try Bar.parse(["--identifier", "not a number"]))
    XCTAssertThrowsError(try Bar.parse(["--identifier", "123.456"]))
  }
}

struct LogLevel: RawRepresentable, CustomStringConvertible {
  var rawValue: String
  var description: String { rawValue }
}

extension LogLevel: LosslessStringConvertible {
  init(_ description: String) {
    self.rawValue = description
  }
}

extension LogLevel: ExpressibleByArgument {}
